#include "SuperString.h"
#include "FSUtils.h"

/*
	The Drag Manager defines a special data flavor for promising file system
	objects. The Promise HFS flavor is used when you wish to create a new 
	file when dragging to the Finder. The flavor consists of an array of the
	following PromiseHFSFlavor structures, with the first entry being the 
	preferred file type you would like to create, and subsequent array 
	entries being file types in descending preference. This structure allows
	you to create the file in your DragSendDataProc, and provide the FSSpec 
	for the new file at that time.

	typedef struct PromiseHFSFlavor {
		OSType			fileType;		// file type
		OSType			fileCreator;	// file creator
		unsigned short	fdFlags;		// Finder flags
		FlavorType 		promisedFlavor;	// FSSpec flavor
	} HFSFlavor;

	"PromiseHFSFlavor.promisedFlavor"
	is the FlavorType of a separate promised flavor to contain the FSSpec 
	for the new file. Call AddDragItemFlavor and promise this separate
	flavor if you wish to create the file in your DragSendDataProc. After
	providing an FSSpec in this flavor, the Finder will move the new file to
	the drop location. If you wish to create the file before the drag and
	provide the FSSpec data up front, create the new file in the Temporary
	Items folder so it does not prematurely appear in an open Finder window.
*/

OSStatus	FSrCreateFromAEDesc(const AEDesc *descRef, FSRef *refP)
{
	OSStatus		err = noErr;
	AEDesc			targetDesc;
	
	ERR(AECoerceDesc(descRef, typeFSRef, &targetDesc));
	
	if (!err) {
		ERR(AEGetDescData(&targetDesc, refP, sizeof(FSRef)));
		ERR2(AEDisposeDesc(&targetDesc));
	}
	
	return err;
}

OSStatus	AEDescCreateFromFSRef(const FSRef &ref, AEDesc *descRef)
{
	OSStatus		err = noErr;

	AEInitializeDescInline(descRef);
	ERR(AECreateDesc(typeFSRef, &ref, sizeof(FSRef), descRef));
	return err;
}


OSErr		FSSpecFromDragRef(DragRef dragRef, Str32 fName, FSSpec *fsSpecP)
{
	OSErr		err = noErr;
	OSErr		err2;
	AEDesc		dropLocDesc, targetDirDesc;
	
	//	get target folder
	if (!err) err = GetDropLocation(dragRef, &dropLocDesc);
	if (!err) {
		if (dropLocDesc.descriptorType != typeAlias) {
			err = paramErr;
		}
		
		if (!err) err = AECoerceDesc(&dropLocDesc, typeFSS, &targetDirDesc);
		if (!err) {
			err = AEGetDescData(&targetDirDesc, fsSpecP, sizeof(FSSpec));
		
			err2 = AEDisposeDesc(&targetDirDesc);
			if (!err) err = err2;
		}
		
		err2 = AEDisposeDesc(&dropLocDesc);
		if (!err) err = err2;
	}
	
	//	descend into folder
	if (!err) err = FSpGetFolderID(fsSpecP, &fsSpecP->parID);
	
	// construct the target FSSpec, verify the name is free...
	if (!err) {
		err = FSMakeFSSpec(
			fsSpecP->vRefNum, fsSpecP->parID, fName, fsSpecP);
		
		if (err == noErr) err = dupFNErr;
		else if (err == fnfErr) err = noErr;
	}
	
	return err;
}

OSErr		DragWentToTrash(DragRef theDrag, Boolean *inTrashB)
{
	OSErr		err;
	AEDesc		descLocation;
	
	*inTrashB = FALSE;
	
	err	= GetDropLocation(theDrag, &descLocation);
	
	if (!err) {
		FSSpec		spec, specTrash;
		AEDesc		resultDesc;
		CInfoPBRec	cpb;
		
		err = AECoerceDesc(&descLocation, typeFSS, &resultDesc );
		if (!err) err = AEGetDescData( &resultDesc, &spec, sizeof(FSSpec) );
		
		if (!err) {
			
			cpb.hFileInfo.ioFDirIndex	= 0;
			cpb.hFileInfo.ioNamePtr		= spec.name;
			cpb.hFileInfo.ioVRefNum		= spec.vRefNum;
			cpb.hFileInfo.ioDirID		= spec.parID;
			err	= PBGetCatInfoSync( &cpb );
		}
		
		if (!err) {
			err = FindFolder(
				kOnAppropriateDisk, kTrashFolderType, 
				true, &specTrash.vRefNum, &specTrash.parID);

			if (
				!err 
				&& (specTrash.vRefNum == cpb.hFileInfo.ioVRefNum)
				&& (specTrash.parID == cpb.hFileInfo.ioDirID)
			) {
				*inTrashB = TRUE;
			}
		}
	}
	
	return err;
}


static pascal Size MinimumBytesForFSSpec(const FSSpec *fss)
{
	// callers can and do assume this does not move memory
	return (Size)(sizeof(*fss) - sizeof(fss->name) + fss->name[0] + 1);
}

OSErr	HFSFlavorDataFromDrag(
	DragReference	theDrag, 
	ItemReference	itemRef, 
	HFSFlavor		*hfsFlavor)
{
	OSErr	err = noErr;
	long	size = sizeof(*hfsFlavor);
	
	err = GetFlavorData(
		theDrag, itemRef, 
		flavorTypeHFS,
		hfsFlavor, &size, 0);
	
	//	see tech note 1085 for explaination of below code
	if (!err) {
		Size	minSize = sizeof(*hfsFlavor) - sizeof(hfsFlavor->fileSpec);
		
		minSize += MinimumBytesForFSSpec(&(hfsFlavor->fileSpec));
		
		if (size < minSize) {
			err = cantGetFlavorErr;
		}
	}
	
	if (!err) err = FSpResolveAlias(&(hfsFlavor->fileSpec));
	
	return err;
}

OSStatus	FSRefVecFromDrag(DragReference theDrag, FSRefVec *refVecP)
{
	OSStatus			err = noErr;
	UInt16				numItems	= 0;
	ItemReference		itemRef;
	HFSFlavor			hfsFlavor;
	FSRef				fsRef;
	
	ERR(CountDragItems(theDrag, &numItems));
	
	loop (numItems) {
		ERR(GetDragItemReferenceNumber(theDrag, _indexS + 1, &itemRef));
		ERR(HFSFlavorDataFromDrag(theDrag, itemRef, &hfsFlavor));
		ERR(FSpMakeFSRef(&hfsFlavor.fileSpec, &fsRef));
		if (!err) refVecP->push_back(fsRef);
	}

	return err;
}

OSErr		AEGet1FSSpec(const AEDescList *docListP, FSSpec *fsSpecP)
{
	OSErr	anErr = noErr;
	long	count;
	
	anErr = AECountItems(docListP, &count);

	if (anErr == noErr) {
		AEKeyword	theKeyword;
		DescType	actualType;
		Size		actualSize;
		
		// Get a pointer to selected file
		anErr = AEGetNthPtr(
			docListP, 1,
			typeFSS, &theKeyword,
			&actualType, fsSpecP,
			sizeof(FSSpec),
			&actualSize);
	}
	
	return anErr;
}

OSStatus	FSrResolveAlias(FSRef *fileRefP, UInt32 mountFlags)
{
	Boolean		wasFolderB, wasAliasB;
	
 	return FSResolveAliasFileWithMountFlags(
 		fileRefP, TRUE, &wasFolderB, &wasAliasB, mountFlags);
}	

OSErr		FSpResolveAlias(
	FSSpec		*fileSpecP)
{
	OSErr			err = noErr;
	Boolean			wasFolderB, wasAliasB;
	
	err = ResolveAliasFile(fileSpecP, TRUE, &wasFolderB, &wasAliasB);
	
	return err;
}

OSErr		FSpFindFolder(
	short		vRefNum, 
	OSType		folderType, 
	FSSpec		*fileSpec)
{
	OSErr		err = noErr;
	FSRef		fsRef;
	
	ERR(FSrFindFolder(
		vRefNum, folderType, &fsRef));
	ERR(FSrMakeFSSpec(fsRef, fileSpec));
	
	return err;
}


OSErr			FSpGetFolderID(const FSSpec *folderSpecP, long *dirID)
{
	OSErr		err			= noErr;
	CInfoPBRec	paramBlock;
	
	memclr(&paramBlock, sizeof(CInfoPBRec));
	
	paramBlock.dirInfo.ioNamePtr		= (unsigned char *)folderSpecP->name;
	paramBlock.dirInfo.ioVRefNum		= folderSpecP->vRefNum;
	paramBlock.dirInfo.ioDrDirID		= folderSpecP->parID;
	
	err = PBGetCatInfo(&paramBlock, FALSE);
	
	if (!err) {
		if (paramBlock.dirInfo.ioFlAttrib & kioFlAttribDirMask == 0) {
			err = errFSNotAFolder;
		} else {
			*dirID = paramBlock.dirInfo.ioDrDirID;
		}
	}
	
	return err;
}

OSErr		FSpCreateFileOrFolder(
	FSSpec		*fileSpecP, 
	OSType		fileType, 
	OSType		fileCreator, 
	Boolean		isFolderB)
{
	OSErr	err;
	
	if (isFolderB) {
		long	newDirID;
		
		err = FSpDirCreate(fileSpecP, smSystemScript, &newDirID);
	} else {
		err = FSpCreate(fileSpecP, fileCreator, fileType, smSystemScript);
	}
	
	return err;
}

OSErr	FSFlushFile(short fileRefNum)
{
	ParamBlockRec	params;
	
	memclr(&params, sizeof(ParamBlockRec));
	
	params.ioParam.ioRefNum = fileRefNum;
	return PBFlushFile(&params, FALSE);
}

OSErr	FSpGetParentFolder(FSSpec *fileSpec, FSSpec *parentSpec)
{
	OSErr	err = FSMakeFSSpec(
		fileSpec->vRefNum, fileSpec->parID, "\p:", parentSpec);

	if (err == dirNFErr || err == fnfErr) {
		*parentSpec = *fileSpec;
		err = noErr;
	}

	return err;	
}

OSErr	FSpGetParentVolume(FSSpec *fileSpec, FSSpec *parentSpec)
{
	OSErr		err = noErr;
	FSRef		volRef;
	
	ERR(FSrGetVolume(fileSpec->vRefNum, &volRef));
	ERR(FSrMakeFSSpec(volRef, parentSpec));
	
/*
	ETR(FSpGetParentFolder(fileSpec, parentSpec));

	if (!FSpEqual(fileSpec, parentSpec)) {
		FSSpec		localPar(*parentSpec);

		ERR(FSpGetParentVolume(&localPar, parentSpec));
	}
*/

	return err;
}

OSStatus		FSGetVolSize(short vRefNum, UInt64 *freeSpaceLL)
{
	OSStatus				err				= noErr;
	FSVolumeInfoParam		volInfoParam;	structclr(volInfoParam);
	FSVolumeInfo			volInfo;		structclr(volInfo);
	
	volInfoParam.ioVRefNum	= vRefNum;
	volInfoParam.whichInfo	= kFSVolInfoSizes;
	volInfoParam.volumeInfo	= &volInfo;

	ERR(PBGetVolumeInfoSync(&volInfoParam));

	if (!err) {
		*freeSpaceLL = volInfo.freeBytes;
	}
	
	return err;
}

OSErr	FSGetVolDriveNum(short vRefNum, short *driveNumS)
{
	OSErr				err;
	HParamBlockRec		paramBlock;
	unsigned char		volName[256];

	memclr(&paramBlock, sizeof(HVolumeParam));

	paramBlock.volumeParam.ioNamePtr	= volName;
	paramBlock.volumeParam.ioVRefNum	= vRefNum;
	
	err = PBHGetVInfo(&paramBlock, FALSE);
	
	if (!err) {
		*driveNumS = paramBlock.volumeParam.ioVDrvInfo;
	}
	
	return err;
}

OSErr	FSpIsFolder(FSSpec *fileSpec, Boolean *isFolderB)
{
	OSErr			err;
	CInfoPBRec		paramBlock;
	
	memclr(&paramBlock, sizeof(CInfoPBRec));

	paramBlock.hFileInfo.ioNamePtr		= fileSpec->name;
	paramBlock.hFileInfo.ioVRefNum		= fileSpec->vRefNum;
	paramBlock.hFileInfo.ioDirID		= fileSpec->parID;
	
	err = PBGetCatInfo(&paramBlock, FALSE);
	
	if (!err) {
		*isFolderB = (Boolean)((paramBlock.hFileInfo.ioFlAttrib & kioFlAttribDirMask) != 0);
	}
	
	return err;
}

OSErr		FSpThisApp(FSSpec *appFSSpec)
{
	ProcessSerialNumber		myProcess;
	ProcessInfoRec			myProcInfo;
	OSErr					err;

	//	ADFS_Log("getting this app's file spec\n");
	err = GetCurrentProcess(&myProcess);
//	if (err) ReportErrorStr(err, "Error getting my process.");

	if (!err) {
		myProcInfo.processInfoLength	= sizeof(ProcessInfoRec);
		myProcInfo.processName			= NULL;
		myProcInfo.processAppSpec		= appFSSpec;

		//	ADFS_Log("getting this app's process info\n");
		err = GetProcessInformation(&myProcess, &myProcInfo);
//		if (err) ReportErrorStr(err, "Error getting process info.");
	}
	
	if (!err) err = FSpGetParentFolder(appFSSpec, appFSSpec);
	if (!err) err = FSpGetParentFolder(appFSSpec, appFSSpec);
	if (!err) err = FSpGetParentFolder(appFSSpec, appFSSpec);
	
	return err;
}

OSErr		FSrLaunchInFinder(const FSRef &fileRef)
{
	OSErr		err = noErr;
	
	if (LSOpenFSRef(&fileRef, NULL) != noErr) {
		err = dsBadLaunch;
	}
	
	return err;
}

OSErr		FSpLaunchInFinder(FSSpec *fileSpecP)
{
	OSErr		err = noErr;
	FSRef		fsRef;
	
	err = FSpMakeFSRef(fileSpecP, &fsRef);
	if (!err) err = FSrLaunchInFinder(fsRef);

	return err;
}

OSErr		FSpGetResEOF(FSSpec *fileSpec, long *resSize)
{
	OSErr			err;
	CInfoPBRec		paramBlock;
	
	memclr(&paramBlock, sizeof(CInfoPBRec));

	paramBlock.hFileInfo.ioNamePtr		= fileSpec->name;
	paramBlock.hFileInfo.ioVRefNum		= fileSpec->vRefNum;
	paramBlock.hFileInfo.ioDirID		= fileSpec->parID;
	
	err = PBGetCatInfo(&paramBlock, FALSE);
	
	if (!err) {
		*resSize = paramBlock.hFileInfo.ioFlRLgLen;
	}
	
	return err;
}

OSErr	FSpGetEOF(FSSpecPtr fileSpec, long *fileSize)
{
	OSErr		err = noErr;
	short		refNum;
	
	err = FSpOpenDF(fileSpec, fsCurPerm, &refNum);
	
	if (!err) {
		OSErr	err2;
		
		err = GetEOF(refNum, fileSize);
		
		err2 = FSClose(refNum);
		
		if (!err) err = err2;
	}
	
	return err;
}

OSErr		FSpSetFileLock(FSSpec *fileSpecP, Boolean lockedB)
{
	OSErr		err = noErr;
	
	if (lockedB) {
		err = FSpSetFLock(fileSpecP);
	} else {
		err = FSpRstFLock(fileSpecP);
	}
	
	return err;
}

OSErr	FSpCreateTempFile(
	short				vRefNum, 
	ConstStr255Param	nameP, 
	OSType				fileCreator, 
	OSType				fileType, 
	FSSpec				*returnSpecP)
{
	OSErr		err = noErr;
	long		folderID;

	//	ADFS_Log("Finding temp items folder\n");
	if (!err) err = FSpFindFolder(
		vRefNum, kTemporaryFolderType, returnSpecP);
	
	
	if (!err) {
		//	ADFS_Log("Getting temp folder ID\n");
		err = FSpGetFolderID(returnSpecP, &folderID);
	}

	if (!err) {
		//	ADFS_Log("Making FSSpec for new temp file\n");
		err = FSMakeFSSpec(
			returnSpecP->vRefNum, folderID, 
			nameP, returnSpecP);

		if (err == fnfErr) {
			err = noErr;
		} else if (err == noErr) {
			//	ADFS_Log("temp file already exists with that name, so "
			//		"attempting to unlock it (so I can delete it)\n");
			
			err	= FSpSetFileLock(returnSpecP, FALSE);
			if (!err) {
				//	ADFS_Log("deleting that file\n");
				err = FSpDelete(returnSpecP);
			}
		}

		if (!err) {
			//	ADFS_Log("Creating new temp file: \n");
			err = FSpCreate(
				returnSpecP, fileCreator, 
				fileType, smSystemScript);
			
			#ifdef kDEBUG
			{
				char		pathAC[kMaxFilePathLen];
				
				if (!err) err = FSpGetFullPath(returnSpecP, kMaxFilePathLen, pathAC);
				if (!err) {
					//	ADFS_Log(pathAC);
					//	ADFS_Log("\n");
				}
			}
			#endif
		}
	}
	
	return err;
}


OSErr			FSpCountFilesInFolder(FSSpec *folderSpecP, unsigned long *numFilesL)
{
	OSErr		err			= noErr;
	CInfoPBRec	pbRec;
	
	memclr(&pbRec, sizeof(CInfoPBRec));
	
	pbRec.dirInfo.ioNamePtr		= folderSpecP->name;
	pbRec.dirInfo.ioVRefNum		= folderSpecP->vRefNum;
	pbRec.dirInfo.ioDrDirID		= folderSpecP->parID;
	
	err = PBGetCatInfo(&pbRec, FALSE);
	
	if (!err) {
		*numFilesL = pbRec.dirInfo.ioDrNmFls;
	}
	
	return err;
}

OSErr			FSpGetIndFileInFolder(
	FSSpec	*folderSpecP, 
	unsigned long	fileIndex, 
	FSSpec	*indFileSpecP)
{
	OSErr		err			= noErr;
	CInfoPBRec	pbRec;
	
	memclr(&pbRec, sizeof(CInfoPBRec));
	pstrcpy(indFileSpecP->name, folderSpecP->name);
	
	pbRec.dirInfo.ioNamePtr		= indFileSpecP->name;
	pbRec.dirInfo.ioVRefNum		= folderSpecP->vRefNum;
	pbRec.dirInfo.ioDrDirID		= folderSpecP->parID;
	
	err = PBGetCatInfo(&pbRec, FALSE);
	
	if (!err) {
		pbRec.dirInfo.ioFDirIndex	= (short)(fileIndex + 1);
		err = PBGetCatInfo(&pbRec, FALSE);
	}

	if (!err) {
		indFileSpecP->vRefNum	= pbRec.dirInfo.ioVRefNum;
		indFileSpecP->parID		= pbRec.dirInfo.ioDrParID;

		if (
			(pbRec.hFileInfo.ioFlFndrInfo.fdFlags & kIsInvisible) != 0
			|| (indFileSpecP->name[1] == '.')
		) {
			err = kErr_FILE_IS_INVISIBLE;
		}
	}
	
	return err;
}

Boolean		FSpEqual(
	const FSSpec	*spec1, 
	const FSSpec	*spec2)
{
	Boolean		sameB = FALSE;
	
	sameB = spec1->vRefNum == spec2->vRefNum
		&& spec1->parID == spec2->parID;
		
	if (sameB) {
		sameB = EqualString(spec1->name, spec2->name, TRUE, TRUE);
	}

	return sameB;
}


OSStatus	LaunchURL(const char *url_AC)
{
	ScCFReleaser<CFURLRef>		url(CFURLCreateWithUTF8(uc(url_AC)));
	
	return LSOpenCFURLRef(url, NULL);
} 

/*

you want: FSrMakeFSSpec
or FSpMakeFSRef

OSErr	FSRefToFSSpec(
	const FSRef	*fsRef, 
	FSSpec		*fsSpec)
{
	return FSGetCatalogInfo(kFSCatInfoNone, NULL, NULL, fsSpec, NULL);
}
*/


OSErr		FSpFindOrCreateFolder(FSSpec &folder, Str255 strP, FSSpec *subDirP)
{
	OSErr		err = noErr;
	long		folderID;
	
	if (!err) err = FSpGetFolderID(&folder, &folderID);
	if (!err) err = FSMakeFSSpec(
		folder.vRefNum, folderID, 
		strP, subDirP);
		
	if (err == fnfErr) err = FSpCreateFileOrFolder(subDirP, 0, 0, TRUE);
	
	return err;
}

OSErr		FSpFindOrCreateFile(
	const FSSpec	&folder, 
	const Str255	strP, 
	OSType			fileType, 
	OSType			fileCreator, 
	FSSpec			*fileP, 
	bool			*newB0)
{
	OSErr		err = noErr;
	long		folderID;
	
	if (!err) err = FSpGetFolderID(&folder, &folderID);
	if (!err) err = FSMakeFSSpec(
		folder.vRefNum, folderID, 
		strP, fileP);

	if (newB0) {
		*newB0 = err == fnfErr;
	}
	
	if (err == fnfErr && fileType != 'xxxx') {	//	kJamsFileType_NO_CREATE) {
		err = FSpCreateFileOrFolder(
			fileP, fileType, fileCreator, FALSE);
	}
	
	return err;
}

/*****************************************************************************/
OSErr		FSrFindFolder(
	short		vRefNum, 
	OSType		folderType, 
	FSRef		*fileRef)
{
	OSErr		err = noErr;
	
	ERR(FSFindFolder(vRefNum, folderType, true, fileRef));
	
	if (err == fnfErr && folderType == kTrashFolderType) {
		err = noErr;
		
		ERR(FSrGetVolume(vRefNum, fileRef));
		ERR(FSrFindOrCreateFolder(*fileRef, "kJams Trash", fileRef));
	}
	
	return err;
}

static	OSErr	FSrObjectExists(
	const FSRef		&parentRef, 
	const UniString	&uniStr, 
	bool			*existsBP, 
	FSRef			*returnRefP0 = NULL)
{
	OSErr			err			= noErr;
	FSRef			tempRef;	if (returnRefP0 == NULL) returnRefP0 = &tempRef;

	*existsBP = false;
	
	err = FSMakeFSRefUnicode(
		&parentRef, uniStr.i_lengthL, uniStr.i_nameP, 
		kTextEncodingDefaultFormat, returnRefP0);
	
	if (err == fnfErr) {
		err = noErr;
	} else if (!err) {
		*existsBP = true;
	}
	
	return err;
}

OSErr	FSrFileExists(
	const FSRef		&parentRef, 
	const UniString	&uniStr, 
	bool			*existsBP, 
	FSRef			*returnRefP0)
{
	FSRef		tempRef;	if (returnRefP0 == NULL) returnRefP0 = &tempRef;
	OSErr		err	= FSrObjectExists(
		parentRef, uniStr, existsBP, returnRefP0);

	if (!err && *existsBP) {
		err = FSrIsFolder(*returnRefP0, existsBP);
		if (!err) {
			*existsBP = !*existsBP;
		}
	}
	
	return err;
}

OSErr	FSrFolderExists(
	const FSRef		&parentRef, 
	const UniString	&uniStr, 
	bool			*existsBP, 
	FSRef			*returnRefP0, 
	bool			resolveB)
{
	FSRef		tempRef;	
	
	if (returnRefP0 == NULL) {
		returnRefP0 = &tempRef;
	}
	
	*existsBP = false;
	
	OSErr		err	= FSrObjectExists(
		parentRef, uniStr, existsBP, returnRefP0);

	if (!err && *existsBP && resolveB) {
		ERR(FSrResolveAlias(returnRefP0));
		ERR(FSrIsFolder(*returnRefP0, existsBP));
	}
	
	return err;
}

OSErr		FSrFindOrCreateFile(
	const FSRef		&folder, 
	const UniString	&uniStr, 
	FSRef			*fileP, 
	OSType			fileType, 
	OSType			fileCreator, 
	bool			*newB0)
{
	OSErr				err			= noErr;
	FSCatalogInfo		catInfo;	structclr(catInfo);
	FileInfo			*fInfoP		= (FileInfo *)&catInfo.finderInfo[0];
	bool				existsB;
	
	fInfoP->fileType	= fileType;
	fInfoP->fileCreator	= fileCreator;

	ERR(FSrObjectExists(folder, uniStr, &existsB, fileP));

    if (!err) {
		if (newB0) {
			*newB0 = !existsB;
		}
    
		if (!existsB) {
			if (fileType != 'xxxx') {	//	kJamsFileType_NO_CREATE) {
				err = FSCreateFileUnicode(
					&folder, uniStr.i_lengthL, uniStr.i_nameP, 
					kFSCatInfoFinderInfo, &catInfo, fileP, NULL);
			} else {
				err = fnfErr;
			}
		}
	}
	
	return err;
}

OSStatus	FSrWriteFile(const FSRef &fileRef, void *dataP, size_t sizeL, bool pictB)
{
	OSStatus	err = noErr;
	short		refNum;
	
	ERR(FSrOpenDF(fileRef, fsWrPerm, &refNum));
	
	if (!err) {
		long int		file_sizeL;
		
		ERR(SetEOF(refNum, 0));

		if (pictB) {
			char	buf[512];	memclr(buf, 512);
			
			file_sizeL = 512;
			ERR(FSWrite(refNum, &file_sizeL, buf));
		}

		file_sizeL = sizeL;
		ERR(FSWrite(refNum, &file_sizeL, dataP));
		(void)FSClose(refNum);
	}
	
	return err;
}

OSErr		FSrFindOrCreateFolder(
	const FSRef		&folder, 
	const UniString	&uniStr, 
	FSRef			*subDirP, 
	bool			*newBP0)
{
	bool	existsB;
	OSErr	err	= FSrFolderExists(folder, uniStr, &existsB, subDirP);
    
	if (!err) {
	
		if (newBP0) {
			*newBP0 = !existsB;
		}
		
		if (!existsB) {
			ERR(FSCreateDirectoryUnicode(
				&folder, uniStr.i_lengthL, uniStr.i_nameP, 
				kFSCatInfoNone, NULL, subDirP, NULL, NULL));
		}
	}
	
	return err;
}

OSErr		AEGet1FSRef(const AEDescList *docListP, FSRef *fsRefP)
{
	FSRefVec	refVec;
	OSErr		err = AEDescListToRefVec(docListP, refVec);
	
	if (!err) {
		if (refVec.size() < 1) {
			err = errAEBadListItem;
		} else {
			*fsRefP = refVec[0];
		}
	}
	
	return err;
}

OSErr	FSrGetParentFolder(const FSRef &fileRef, FSRef *parentRefP)
{
	FSCatalogInfo		catInfo;	structclr(catInfo);

	OSErr	err = FSGetCatalogInfo(
		&fileRef, 
		kFSCatInfoParentDirID, 
		&catInfo, 
		NULL, 
		NULL, 
		parentRefP);
	
	if (catInfo.parentDirID == fsRtParID) {
		*parentRefP = fileRef;
	}

	return err;	
}

bool	FSrEqual(
	const FSRef		&ref1, 
	const FSRef		&ref2)
{
	return FSCompareFSRefs(&ref1, &ref2) == noErr;
}

OSErr	FSrGetParentVolume(const FSRef &fileRef, FSRef *parentRefP)
{
	OSErr		err;

	err = FSrGetParentFolder(fileRef, parentRefP);
	if (!err) {
		if (!FSrEqual(fileRef, *parentRefP)) {
			FSRef		localPar(*parentRefP);

			err = FSrGetParentVolume(localPar, parentRefP);
		}
	}

	return err;
}

OSStatus	FSrIsVolume(const FSRef &fileRef, bool *isVolBP)
{
	OSStatus		err = noErr;
	FSRef			volRef;
	
	ERR(FSrGetParentVolume(fileRef, &volRef));
	
	if (!err) {
		*isVolBP = FSrEqual(fileRef, volRef);
	}
	
	return err;
}


OSErr		FSrThisApp(FSRef *appRef)
{
	FSSpec		spec;
	OSErr		err = FSpThisApp(&spec);
	
	if (!err) err = FSpMakeFSRef(&spec, appRef);
	return err;
}

OSErr	FSrGetEOF(const FSRef &fileRef, UInt64 *fileSize)
{
	FSCatalogInfo		catInfo;
	OSErr				err = FSGetCatalogInfo(
		&fileRef, kFSCatInfoDataSizes, &catInfo, NULL, NULL, NULL);
	
	*fileSize = catInfo.dataLogicalSize;
	return err;
}

OSErr	FSrSetEOF(const FSRef &fileRef, UInt64 fileSizeLL)
{
	OSStatus	err = noErr;
	short		refNum;
	
	ERR(FSrOpenDF(fileRef, fsWrPerm, &refNum));
	if (!err) {
		ERR(FSSetForkSize(refNum, fsFromStart, fileSizeLL));
		ERR2(FSClose(refNum));
	}
	
	return err;
}

#define		loop64(_n)					for (UInt64 _indexS = 0, _maxS = (_n); _indexS < _maxS; _indexS++)

OSStatus		FSrPadFile(const FSRef &fileRef, UInt64 loopNum, UInt64 sizeL, bool zeroB)
{
	OSStatus	err = noErr;
	
	if (zeroB) {
		short			fileRefNum;
		
		ERR(FSrOpenDF(fileRef, fsWrPerm, &fileRefNum, true));
		if (!err) {
			long			writeSizeL;
			CharVec			sectorVec(sizeL);
		
			memclr(&sectorVec[0], sectorVec.size());
			ERR(SetFPos(fileRefNum, fsFromLEOF, 0));
			
			loop64 (loopNum) {
				writeSizeL = sectorVec.size();
				ERR(FSWrite(fileRefNum, &writeSizeL, &sectorVec[0]));
				
				if (writeSizeL != sectorVec.size()) {
					ERR(notExactSizeErr);
				}
					
				if (err) break;
			}

			ERR2(FSClose(fileRefNum));
		}
	} else {
		UInt64		curSizeL;

		ERR(FSrGetEOF(fileRef, &curSizeL));
		curSizeL = (loopNum * sizeL) + curSizeL;
		ERR(FSrSetEOF(fileRef, curSizeL));
	}
	
	return err;
}

//	FSpMakeFSRef
OSErr			FSrMakeFSSpec(
	const FSRef		&fileRef, 
	FSSpec			*specP)
{
	return FSGetCatalogInfo(&fileRef, kFSCatInfoNone, NULL, NULL, specP, NULL);
}

OSErr		FSrCountFilesInFolder(const FSRef &folderRef, unsigned long *numFilesL)
{
	FSSpec		folderSpec;
	OSErr		err = FSrMakeFSSpec(folderRef, &folderSpec);
	
	if (!err) err = FSpCountFilesInFolder(&folderSpec, numFilesL);
	return err;
}

OSErr		FSrGetIndFileInFolder(
	const FSRef		&folderRef, 
	unsigned long	fileIndex, 
	FSRef			*indFileRefP)
{
	FSSpec		folderSpec;
	OSErr		err = FSrMakeFSSpec(folderRef, &folderSpec);
	
	if (!err) err = FSpGetIndFileInFolder(&folderSpec, fileIndex, &folderSpec);
	if (!err) err = FSpMakeFSRef(&folderSpec, indFileRefP);
	return err;
}

OSStatus	FSrGetExtensionType(const FSRef &fileRef, OSType *typeP, SuperString *nameP0)
{
	OSStatus		err = noErr;
	SuperString		nameStr;
	
	if (nameP0 == NULL) {
		nameP0 = &nameStr;
	}
	
	*typeP = 0;

	ERR(FSrGetName(fileRef, nameP0));
	if (!err) {
		*typeP = GetExtension(nameP0->c_str());
	}
	
	return err;
}

OSErr		FSrGetName(const FSRef &fileRef, SuperString *superStrP)
{
	HFSUniStr255	nameRec;
	OSErr			err = FSGetCatalogInfo(
		&fileRef, kFSCatInfoNone, NULL, &nameRec, NULL, NULL);
	
	if (!err) {
		superStrP->Set(nameRec);
	}
	
	return err;
}

OSErr		FSrCopyNameRef(const FSRef &fileRef, CFStringRef *nameRefP)
{
	OSErr			err = noErr;
	SuperString		superStr;
		
	ERR(FSrGetName(fileRef, &superStr));
	if (!err) {
		CFRetain(*nameRefP = superStr);
	}
	
	return err;
}

OSErr		FSrGetName(const FSRef &fileRef, std::string *stdStringP)
{
	OSErr				err = noErr;
	SuperString			str;
	
	ERR(FSrGetName(fileRef, &str));
	if (!err) {
		*stdStringP = str.std();
	}
	
	return err;
}

OSErr		FSrOpenDF_LowLev(
	const FSRef		&fileRef, 
	SInt8			permissions,
	short *			refNum)
{
	HFSUniStr255	uniStr;
	OSErr			err = FSGetDataForkName(&uniStr);
	
	ERR(FSOpenFork(
		&fileRef, uniStr.length, 
		uniStr.unicode, permissions, refNum));
	
	if (err) {
		*refNum = 0;
	}
	
	return err;
}

OSErr		FSrOpenDF(
	const FSRef		&fileRef, 
	SInt8			permissions,
	short *			refNum, 
	bool			keepTryingB)
{
	OSErr	err = noErr;
	
	ERR(FSrOpenDF_LowLev(fileRef, permissions, refNum));
	
	return err;
}

/*******************************************/

#include <numeric>

static	void	TruncAtSlash(ustring &str, short slashS)
{
	UTF8Char	*chP = &str[1];
	
	loop (slashS + 1) {
		chP = (UTF8Char *)strchr((const char *)++chP, '/');
	}
		
	if (chP && chP < &*str.end()) {
		str.resize(std::distance(&str[0], chP));
	}
}

class CAddSlashes {

	public: short	operator()(short prevS, const UTF8Char &ch) {
		return prevS + (ch == '/');
	}
	
};

static	short	count_slashes(const ustring &str)
{
	return std::accumulate(str.begin(), str.end(), (short)0, CAddSlashes());
}

OSStatus		FSrResolveAsFarAsYouCan(
	const ustring	&utf8Path, 
	FSRef			*fileRef)
{
	OSStatus	err = noErr;
	ustring		path(utf8Path);
	
	ERR(FSrFixVolumePath(&path));
	
	if (!err) {
		ustring		part;
		FSRef		curRef;

		err = fnfErr;

		loop (count_slashes(path)) {
			part = path;
			TruncAtSlash(part, _indexS);
			
			if (FSPathMakeRef(part.c_str(), &curRef, NULL) == noErr) {
				*fileRef = curRef;
				err = noErr;
			} else {
				break;
			}
		}
	}
	
	return err;
}

OSStatus		FSrCreateFromPath(
	const ustring	&utf8Path, 
	FSRef			*fileRef)
{
	OSStatus		err = noErr;
	
	//ASSERT(fileRef);

	if (utf8Path.empty()) {
		err = emptyPathErr;
	} else {
		ERR(FSPathMakeRef(utf8Path.c_str(), fileRef, NULL));
		
		#ifdef kDEBUG
			if (err) {
				ustring		part;
				
				loop (count_slashes(utf8Path)) {
					part = utf8Path;
					TruncAtSlash(part, _indexS);
					
					err = FSPathMakeRef(part.c_str(), fileRef, NULL);
					Logf("%s: %s\n", err ? "err" : " OK", (const char *)part.c_str());
					if (err) break;
				}
			}
		#endif
		
		if (err) {
			err = noErr;

			SuperString							superStr(utf8Path.c_str());
			ScCFReleaser<CFMutableStringRef>	str(CFStringCreateMutableCopy(NULL, 0, superStr));
			
			CFStringNormalize(str, kCFStringNormalizationFormD);

			SuperString		super2(str);
			
			err = FSPathMakeRef(super2, fileRef, NULL);
		}
	}
	
	return err;
}

/*
OSStatus		FSrCreateFromPath(
	const char *	path, 
	FSRef			*fileRef)
{
	SuperString		superStr(path);
	const UInt8		*uniZ = superStr;
	OSStatus		err = FSPathMakeRef(uniZ, fileRef, NULL);
	
	if (err) err = FSPathMakeRef((UInt8 *)path, fileRef, NULL);
	
	if (err) {
		ScCFReleaser<CFMutableStringRef>	str(CFStringCreateMutableCopy(NULL, 0, superStr));
		
		CFStringNormalize(str, kCFStringNormalizationFormD);

		SuperString		super2(str);
		
		err = FSPathMakeRef(super2, fileRef, NULL);
	}
	
	return err;
}
*/

OSStatus		FSrGetPath(
	const FSRef		&fileRef, 
	ustring			*pathP)
{
	OSStatus		err = noErr;
	UCharVec		utfVec(kMaxFilePathLen * 2);
	
	pathP->clear();
	ERR(FSRefMakePath(&fileRef, &utfVec[0], utfVec.size()));

	if (!err) {
		*pathP = &utfVec[0];
	}

	return err;
}

OSStatus		FSrGetPath(
	const FSRef		&fileRef, 
	SuperString		*superP)
{
	ustring			str;
	OSStatus		err = FSrGetPath(fileRef, &str);
	
	if (!err) {
		superP->Set(str);
	}
	
	return err;
}

/*
OSStatus		FSrGetPathStd(
	const FSRef		&fileRef, 
	std::string		&path)
{
	ustring			buf8;
	OSStatus		err = FSrGetPath(fileRef, &buf8);
	
	if (!err) {
		ScCFReleaser<CFStringRef>	ref = CFStringCreateWithCu(buf8.c_str());
		
		CopyCFStringToStd(ref, path);
	}
	
	return err;
}

OSStatus		FSrGetPathC(
	const FSRef		&fileRef, 
	UInt32			maxPathSize, 
	char *			path)
{
	std::string		str;
	OSStatus		err = FSrGetPathStd(fileRef, str);
	
	if (!err) {
		strncpy(path, str.c_str(), maxPathSize);
	}
	
	return err;
}
*/
/*******************************************/
OSErr		FSrIsFolder(const FSRef &fileSpec, bool *isFolderB)
{
	FSSpec		folderSpec;
	Boolean		is_folderB;
	OSErr		err = FSrMakeFSSpec(fileSpec, &folderSpec);
	
	if (!err) err = FSpIsFolder(&folderSpec, &is_folderB);
	*isFolderB = !!is_folderB;
	return err;
}

OSErr	FSrGetVolume(FSVolumeRefNum vRefNum, FSRef *volRefP, char *pathZ, bool *is_cdB)
{
	OSErr	err = noErr;
	
	ERR(FSGetVolumeInfo(
		vRefNum, 0, NULL, kFSVolInfoNone, NULL, NULL, volRefP));
	
	if (pathZ || is_cdB) {
		ERR(FSVolRefNumToBSD(vRefNum, pathZ, is_cdB));
	}

	return err;
}

OSStatus		FSrGetVolume(
	const SuperString	&searchName, 
	FSRef				*volRefP0, 
	ustring				*pathP0)
{
	OSStatus		err = noErr;
	UInt32			indexL = 0;
	FSRef			volRef;
	HFSUniStr255	volName;
	
	do {
		err = FSGetVolumeInfo(
			kFSInvalidVolumeRefNum,
			indexL + 1,
			NULL, 
			kFSVolInfoNone, 	//	kFSVolInfoFinderInfo | kFSVolInfoFlags | kFSVolInfoFSInfo,
			NULL,
			&volName,
			&volRef);
		
		if (!err) {
			if (SuperString(volName) == searchName) {
				if (volRefP0) {
					*volRefP0 = volRef;
				}
				
				if (pathP0) {
					ERR(FSrGetPath(volRef, pathP0));
				}
				break;
			}
			
			indexL++;
		}
	} while (!err);
	
	return err;
}

/*
Rather than 'attempting' to convert to an HFS path, CFURL will do it for you (admittedly this is not immediately obvious from docs). Typed in Mail:

CFURL url = CFURLCreateWithFileSystemPath( NULL,
			     		posixPath,
			     		kCFURLPOSIXPathStyle,
			     		&isDirectory );
CFStringRef hfsPath = CFURLCopyFileSystemPath( url, kCFURLHFSPathStyle );


Then if needed grab the individual components with something like
CFArrayRef pathComponents = CFStringCreateArrayBySeparatingStrings( NULL, hfsPath, CFSTR( ":" ) );

	if you don't pass an *fsRef, then pathP is updated to new string
	if you DO pass fsRef, then pathP does not get changed
*/
static	OSStatus	p_FSrResolveVolumePath(ustring *pathP, bool upCaseB)
{
	OSStatus		err = noErr;
	
	if (strncmp((const char *)pathP->c_str(), "/Volumes/", 9) == 0) {
		const UTF8Char	*vol_begin	= &pathP->c_str()[9];
		const UTF8Char	*vol_end	= (const UTF8Char *)strchr((const char *)vol_begin, '/');
		ustring			volStr;		volStr.assign(vol_begin, vol_end);

		if (upCaseB) {
			UpperString(volStr);
		}

		ERR(FSrGetVolume(volStr, NULL, &volStr));
		
		if (!err) {
			UTF8Char		newPathAC[256];
			
			sprintf((char *)newPathAC, "%s%s", (char *)volStr.c_str(), (char *)vol_end);
			*pathP = newPathAC;
		}
	}
	
	return err;
}

OSStatus	FSrResolveVolumePath(ustring *pathP)
{
	OSStatus		err = noErr;
	FSRef			fsRef;
	
	ERR(FSrCreateFromPath(*pathP, &fsRef));
	if (err) err = p_FSrResolveVolumePath(pathP, false);
	if (err) err = p_FSrResolveVolumePath(pathP, true);

	return err;
}

OSStatus	FSrFixVolumePath(ustring *pathP, FSRef *fsRefP0)
{
	OSStatus		err = noErr;
	ustring			newPath(*pathP);
//	FSRef			fsRef;
	
	ERR(FSrResolveVolumePath(&newPath));

	if (err) {
	//	Logf("er2: %s\n", (const char *)newPath.c_str());
	} else {
		if (fsRefP0) {
			ERR(FSrCreateFromPath(newPath, fsRefP0));
		} else if (newPath != *pathP) {
			*pathP = newPath;
		}
	}
	
	return err;
}

OSErr		FSrCopyVolumeName(FSVolumeRefNum vRefNum, CFStringRef *stringRefP)
{
	FSRef	fsRef;
	OSErr	err = noErr;
	
	ERR(FSrGetVolume(vRefNum, &fsRef));
	ERR(FSrCopyNameRef(fsRef, stringRefP));
	return err;
}

OSStatus		FSrGetVolumeName(FSVolumeRefNum vRefNum, SuperString *strP)
{
	OSStatus					err = noErr;
	ScCFReleaser<CFStringRef>	volumeNameRef;

	ERR(FSrCopyVolumeName(
		vRefNum, volumeNameRef.AddressOf())); 
	
	if (!err) {
		strP->Set(volumeNameRef);
	}
	
	return err;
}

OSErr		AEDescListToRefVec(const AEDescList *docListP, FSRefVec	&refVec)
{
	OSErr	err = noErr;
	long	count;
	
	err = AECountItems(docListP, &count);

	if (err == noErr) {
		AEKeyword	theKeyword;
		DescType	actualType;
		Size		actualSize;
		
		refVec.resize(count);
		
		loop_reverse (count) {
			err = AEGetNthPtr(
				docListP, _indexS + 1,
				typeFSRef, &theKeyword,
				&actualType, &refVec[_indexS],
				sizeof(FSRef),
				&actualSize);
			
			if (err) break;
		}
	}
	
	return err;
}

OSStatus		FSrGetSizeVec(const FSRefVec &refVec, UInt32Vec *sizeVecP, UInt32 multipleL)
{
	OSStatus	err = noErr;
	UInt64		sizeL;

	loop (refVec.size()) {
		ERR(FSrGetEOF(refVec[_indexS], &sizeL));
		if (err) break;

		if (multipleL) {
			sizeL += (multipleL - 1);
			sizeL /= multipleL;
		}

		sizeVecP->push_back(sizeL);
	}
	
	return err;
}

OSErr		FSrGetCatInfoBulk(
	const FSRef		&parFolderRef, 
	FSRefVec		&refVec)
{
	OSErr				err = noErr;
	FSIterator			fsIterator;
	ItemCount			objectsSoFarL		= 0;
	const ItemCount		objectsPerRequest	= 100;
	ItemCount			objectsReturnedL;
	
	err = FSOpenIterator(&parFolderRef, kFSIterateFlat, &fsIterator);

	if (!err) {

		while (!err) {
			refVec.resize(objectsSoFarL + objectsPerRequest);
			
			err = FSGetCatalogInfoBulk(
				fsIterator, objectsPerRequest, &objectsReturnedL, NULL, 
				kFSCatInfoNone, NULL, &refVec[objectsSoFarL], NULL, NULL);
			
			objectsSoFarL += objectsReturnedL;

			if (err == errFSNoMoreItems) {
				refVec.resize(objectsSoFarL);
			}
		}

		err = FSCloseIterator(fsIterator);
	}
	
	return err;
}

OSErr		FSrCreateURL(
	const FSRef		&fileRef, 
	CFURLRef		*urlP)
{
	OSErr			err = noErr;
	SuperString		pathStr;
	
	ERR(FSrGetPath(fileRef, &pathStr));

	if (!err) {
		*urlP = CFURLCreateWithFileSystemPath(
			kCFAllocatorDefault, pathStr, kCFURLPOSIXPathStyle, NULL);

/*		*urlP = CFURLCreateFromFileSystemRepresentation(
			kCFAllocatorDefault, 
			uniStr.i_nameP, uniStr.i_lengthL, false);
*/		
		err = *urlP == NULL;
	}
	
	return err;
}

OSErr		FSrCreateFromURL(
	CFURLRef		url, 
	FSRef			*fileRefP)
{
	OSErr			err = noErr;
	UCharVec		utfVec(kMaxFilePathLen * 2);
	bool			successB = CFURLGetFileSystemRepresentation(
		url, true, &utfVec[0], utfVec.size());

	if (successB == false) {
		err = noPathMappingErr;
	}
	
	if (!err) {
		ustring			utfStr(&utfVec[0]);

		ERR(FSrCreateFromPath(utfStr, fileRefP));
	}
	
	return err;
}


OSErr	FSrGetVRefNum(const FSRef &fileRef, FSVolumeRefNum *vRefNumP)
{
	OSErr			err = noErr;
	FSSpec			fsSpec;
	
	ERR(FSrMakeFSSpec(fileRef, &fsSpec));

	if (!err) {
		*vRefNumP = fsSpec.vRefNum;
	}
	
	return err;
}

static	OSStatus	CB_S_DeleteFile_Completion(void *dataP, void *fork_dataP, OSStatus err)
{
	std::auto_ptr<FSRef>	fileRefP((FSRef *)dataP);
	return err;
}

static	OSStatus	CB_S_DeleteFile(void *dataP, void *fork_dataP, OSStatus err)
{
//	UInt32				delID = UInt32(fork_dataP);
	FSRef				*fileRefP((FSRef *)dataP);
//	short				countS = 0;
	bool				doneB = false;
	SuperString			nameStr, folderStr, objType;
	std::string			idStr;	//	= ULong_To_Hex(delID);
	bool				isFolderB;

	ERR(FSrIsFolder(*fileRefP, &isFolderB));

/*	if (gApp->Logging()) {
		FSRef	folderRef;
		
		ERRL(FSrIsFolder(*fileRefP, &isFolderB), "determining folderness");
		objType.Set(isFolderB ? "Folder" : "File");
		ERRL(FSrGetName(*fileRefP, &nameStr), "Getting file name for deletion");
		ERRL(FSrGetParentFolder(*fileRefP, &folderRef), "getting delete file's parent folder");
		ERRL(FSrGetName(folderRef, &folderStr), "Getting file name for deletion");
	}
*/

	if (!err) do {
		
		if (isFolderB) {
			ERR(FSDeleteObject(fileRefP));
		} else {
			ustring			str;
			
			ERR(FSrGetPath(*fileRefP, &str));
			
			if (unlink((const char *)str.c_str()) == -1) {
				ERR(errno);
			}
			
			if (err) {
			//	Logf("Unix Error: %s: %s", UnixErrStr("deleting file", err).utf8Z(), (const char *)str.c_str());
			}
		}

doneB = true;
/*
		doneB = err == noErr;
		
		Logf(SuperString("Attempting %s deletion: %s: %s/%s: %s\n").utf8Z(), 
			objType.utf8Z(), 
			idStr.c_str(), 
			folderStr.utf8Z(), 
			nameStr.utf8Z(), 
			doneB ? "Success" : "Failed");
		if (!doneB) {
			err = noErr;
			MPDelay(durationSecond);
			
			if (countS == 0) {
				
				if (isFolderB) {
					FSRef	dsStoreRef;
					
					ERR(FSrFileExists(
						*fileRefP, ".DS_Store", 
						&isFolderB, &dsStoreRef));
					
					if (!err && isFolderB) {
						ERR(FSDeleteObject(&dsStoreRef));
					}
				}
				
				err = noErr;
			}
		}
		
		if (!doneB && ++countS == 5) {
			Logf(SuperString("Gave up, moving %s %s to trash instead\n").utf8Z(), objType.utf8Z(), idStr.c_str());
			(void)FSrMoveToTrash(fileRefP);
			doneB = true;
		}
*/
	} while (!doneB);
	
	return err;
}

//static	CMutex_ThreadQue	*S_delQueP	= NULL;
//static	UInt32				S_deleteID	= 0;

void	FSUtils_PostDispose()
{
//	delete S_delQueP;
//	S_delQueP = NULL;
}

bool		FSrCanWriteToFolder(const FSRef &folderRef)
{
	OSStatus	err = noErr;
	FSRef		testFile;
	
	ERR(FSrFindOrCreateFile(folderRef, CFSTR("write_test"), &testFile));
	bool	successB = err == noErr;
	ERR(FSrDeleteFile(&testFile, true));
	
	return successB;
}

OSStatus	FSrDeleteFile(FSRef *fileRefP, bool trashFirstB)
{
	OSStatus		err		= noErr;
/*
	UInt32			delID	= ++S_deleteID;

	if (S_delQueP == NULL) {
		S_delQueP = new CMutex_ThreadQue("Deletion Que");
	}
	
	if (0) {	//	gApp->Logging()) {
		SuperString		nameStr;
		bool			isFolderB;

		ERR(FSrIsFolder(*fileRefP, &isFolderB));
		ERRL(FSrGetName(*fileRefP, &nameStr), "Getting file name for deletion que");
		Logf(SuperString("Queing %s for deletion: %s: %s\n").utf8Z(), 
			isFolderB ? "Folder" : "File", 
			ULong_To_Hex(delID).c_str(), nameStr.utf8Z());

		err = noErr;
	}
*/
	
	if (trashFirstB) {
		ERR(FSrMoveToTrash(fileRefP));
	}
	
	err = CB_S_DeleteFile(fileRefP, NULL, noErr);
	
/*
	CT_QueElement	elem(
		CB_S_DeleteFile, CB_S_DeleteFile_Completion, 
		new FSRef(*fileRefP), (void *)delID);
	
	S_delQueP->push_back(elem);
*/
	return err;
}

OSErr	FSrMoveToTrash(FSRef *fileRefP)
{
	OSErr			err = noErr;
	FSRef			parentRef;
	FSRef			trashRef;
	short			vRefNum;
	SuperString		nameStr;
	bool			changedB, renameB = false;
	
	ERRL(FSrGetVRefNum(*fileRefP, &vRefNum), "getting vRefNum for trashing");
	ERRL(FSrFindFolder(vRefNum, kTrashFolderType, &trashRef), "finding trash folder");

	ERRL(FSrGetName(*fileRefP, &nameStr), "getting file name for trashing");
	ERRL(FSrGetParentFolder(*fileRefP, &parentRef), "getting parent folder for trashing");
	
	if (!FSrEqual(parentRef, trashRef)) {
		do {		
			ERRL(FSrUniqueName(trashRef, &nameStr, &changedB), "making name unique for trashing");
		
			if (changedB) {
				renameB = true;
				ERRL(FSrUniqueName(parentRef, &nameStr, &changedB), "making name unique for trashing (2)");
			}
		} while (changedB);
		
		if (renameB) {
			ERRL(FSrRename(fileRefP, nameStr), "renaming for trashing");
		}

		ERRL(FSMoveObject(fileRefP, &trashRef, fileRefP), "moving to trash");
	}

	return err;
}

OSErr	FSrOnSameVolume(const FSRef &ref1, const FSRef &ref2, bool *sameB)
{
	FSVolumeRefNum	vol1, vol2;
	OSErr			err = noErr;
	
	*sameB = false;

	ERR(FSrGetVRefNum(ref1, &vol1));
	ERR(FSrGetVRefNum(ref2, &vol2));
	
	if (!err) {
		*sameB = vol1 == vol2;
	}

	return err;
}


/*
 ioNamePtr
	On input, a pointer to the name of the source file.

 ioVRefNum
	 On input, the volume reference number or drive number for the volume containing 
	 the source file. Pass 0 for the default volume.

 ioDirID
	On input, the directory ID of the source directory.


=====================
 ioDstVRefNum
	On input, the reference number or drive number of the destination volume. 
	Pass 0 for the default volume.

 ioNewName
	On input, a pointer to the partial pathname for the destination directory. 
	If ioNewName is NULL, the destination directory is the directory having the 
	ID specified in the ioNewDirID field.

 ioCopyName
	On input, a pointer to the files new name. The string pointed to by this field must 
	be a filename, not a partial pathname. If you do not wish to rename the file, pass 
	NULL in this field.

 ioNewDirID
	On input, if the ioNewName field is NULL, the directory ID of the destination 
	directory. If ioNewName is not NULL, the parent directory ID of the destination directory.


//	this only works for remote file servers

OSErr	FSCopyObject(
	const FSRef *  ref,
	const FSRef *  destDirectory,
	FSRef *        newRef0)
{
	HParamBlockRec		paramRec;	structclr(paramRec);
	FSSpec				refSpec;
	OSErr				err;
	Str255				nameStr;
	
	err = FSrMakeFSSpec(*ref, &refSpec);
	
	if (!err) {
		memcpy(nameStr, refSpec.name, refSpec.name[0] + 1);
		
		paramRec.copyParam.ioNamePtr	= nameStr;
		paramRec.copyParam.ioVRefNum	= refSpec.vRefNum;
		paramRec.copyParam.ioDirID		= refSpec.parID;
	
		err = FSrMakeFSSpec(*destDirectory, &refSpec);
	}

	if (!err) {
		paramRec.copyParam.ioDstVRefNum	= refSpec.vRefNum;
		if (!err) err = FSpGetFolderID(&refSpec, &paramRec.copyParam.ioNewDirID);
		if (!err) err = PBHCopyFileSync(&paramRec);
	}
 	
 	if (!err && newRef0) {
		
		if (!err) err = FSMakeFSSpec(
			refSpec.vRefNum, 
			paramRec.copyParam.ioNewDirID, 
			nameStr, &refSpec);
		
 		if (!err) err = FSpMakeFSRef(&refSpec, newRef0);
 	}
 	
 	return err;
}

*/


#define		kCopyBlockSize		kMegaByte

OSErr	FSrCopyFileToFolder(const FSRef &dstFold, FSRef *srcFileP, SuperString *destNameP0)
{
	OSErr			err	= noErr;
	CharVec			buf(kCopyBlockSize);
	FSRef			dstFile;
	FSSpec			fileSpec;
	SuperString		nameStr;
	FInfo			fInfo;
	short			srcRefNum, dstRefNum;
	
	ERR(FSrMakeFSSpec(*srcFileP, &fileSpec));
	ERR(FSpGetFInfo(&fileSpec, &fInfo));
	
	if (destNameP0) {
		nameStr.Set(*destNameP0);
	} else {
		ERR(FSrGetName(*srcFileP, &nameStr));
	}
	
	if (!err) {
		ERR(FSrFindOrCreateFile(
			dstFold, nameStr, &dstFile, 
			fInfo.fdType, fInfo.fdCreator));
	}
	
	ERR(FSrOpenDF(*srcFileP, fsRdPerm, &srcRefNum));
	
	if (!err) {
		ERR(FSrOpenDF(dstFile, fsWrPerm, &dstRefNum));
		ERRL(SetEOF(dstRefNum, 0), "setting EOF to zero");
		
		if (!err) {
			long		numBytes = kCopyBlockSize;
			
			while (!err && numBytes) {
				ERR(FSRead(srcRefNum, &numBytes, &buf[0]));

				if (err == eofErr) err = noErr;

				ERR(FSWrite(dstRefNum, &numBytes, &buf[0]));
			}
			
			ERR2(FSClose(dstRefNum));
		}

		ERR2(FSClose(srcRefNum));
	}
	
	if (!err) {
	//	(void)(CMeta_Spotlight::Copy(*srcFileP, dstFile));
		*srcFileP = dstFile;
	}
	
	return err;
}

OSErr	FSrCopyToFolder(const FSRef &dstFold, FSRef *srcFileP)
{
	OSErr		err	= noErr;
	bool		isFolderB;

	ERR(FSrIsFolder(*srcFileP, &isFolderB));
	
	if (!err) {
		
		if (isFolderB) {
			FSRef						folderRef;
			ScCFReleaser<CFStringRef>	stringRef;
			
			ERR(FSrCopyNameRef(*srcFileP, stringRef.AddressOf()));
			ERR(FSrFindOrCreateFolder(dstFold, (CFStringRef)stringRef, &folderRef));
		
			FSRefVec		refVec;

			ERR(FSrGetCatInfoBulk(*srcFileP, refVec));
			
			loop (refVec.size()) {
				ERR(FSrCopyToFolder(folderRef, &refVec[_indexS]));
			}
			
			if (!err) {
				*srcFileP = folderRef;
			}
		} else {
			ERR(FSrCopyFileToFolder(dstFold, srcFileP));
		}
	}
	
	return err;
}

OSStatus	FSrMoveFile(const FSRef &dstFold, FSRef *srcFileP)
{
	OSStatus	err = noErr;
	bool		sameB;
	
	ERR(FSrOnSameVolume(dstFold, *srcFileP, &sameB));
	
	if (!err) {
		if (sameB) {
			ScCFReleaser<CFStringRef>	nameStrRef;
			FSRef						dstFile;
			bool						existsB;

			ERR(FSrCopyNameRef(*srcFileP, nameStrRef.AddressOf()));
			ERR(FSrFileExists(
				dstFold, nameStrRef.Get(), 
				&existsB, &dstFile));
				
			if (existsB) {
				if (!FSrEqual(dstFile, *srcFileP)) {
					ERR(FSrMoveToTrash(&dstFile));
					ERR(FSrDeleteFile(&dstFile));
				}
			}

			ERRL(FSMoveObject(srcFileP, &dstFold, srcFileP), "moving to destination");
		} else {
			FSRef		origFile = *srcFileP;
			
			ERR(FSrCopyToFolder(dstFold, srcFileP));
			ERR(FSrDeleteFile(&origFile));
		}
	}
	
	return err;
}

OSStatus		FSrLaunchWithApp(
	const FSRef &fileRef, 
	const FSRef &appRef, 
	bool		asyncB)
{
	LSLaunchFSRefSpec		spec;	structclr(spec);
	
	spec.appRef			= &appRef;
	spec.numDocs		= 1;
	spec.itemRefs		= &fileRef;
	spec.launchFlags	= kLSLaunchDontSwitch | kLSLaunchNoParams;
	
	return LSOpenFromRefSpec(&spec, NULL);
}

/*************************************************************/
// Search through the current process list to find the given application. See
// Using the Process Manager for a similar way of doing this.
typedef struct {
	FSSpec				fsSpec;
	Str255				name;
	ProcessInfoRec		infoRec;
} Net_ProcRec;

static	void	Net_ClearProcRec(
	Net_ProcRec		*procRecP)
{
	memset(procRecP, sizeof(Net_ProcRec), 0);

	procRecP->infoRec.processInfoLength		= sizeof(ProcessInfoRec);
	procRecP->infoRec.processName			= (StringPtr)&procRecP->fsSpec;
	procRecP->infoRec.processAppSpec		= &procRecP->fsSpec;
}

static	OSErr	Net_GetProcess(
	OSType			typeToFind, 
	OSType			creatorToFind,
	Net_ProcRec		*procRecP)
{
	ProcessSerialNumber		psn		= { kNoProcess, kNoProcess };
	Boolean					doneB	= FALSE;
	OSErr					err		= noErr;

	Net_ClearProcRec(procRecP);

	do {
		err = GetNextProcess(&psn);
		if (!err) err = GetProcessInformation(&psn, (ProcessInfoRecPtr)&procRecP->infoRec);
		if (!err) {
			doneB = procRecP->infoRec.processSignature == creatorToFind
				&& procRecP->infoRec.processType == typeToFind;
		}
	} while (!err && !doneB);

	return err;
}

static	OSErr	FindAProcess(
	OSType					typeToFind, 
	OSType					creatorToFind,
	ProcessSerialNumberPtr	procNumP)
{
	Net_ProcRec		procRec;
	OSErr			err = Net_GetProcess(typeToFind, creatorToFind, &procRec);
	
	if (!err) {
		*procNumP = procRec.infoRec.processNumber;
	}
	
	return err;
}

#define kFinderSig			'FNDR'
#define kSystemType			'MACS'

static OSErr		FSrRevealInFinder2(const FSRef &ref)
{
	OSErr					err = noErr;
	FSSpec					fileSpec;
	AppleEvent				aeEvent;		// the event to create;
	AEDesc					myAddressDesc, aeDirDesc, listElem, fileList;
	FSSpec					dirSpec;
	AliasHandle				dirAlias, fileAlias;	// alias of the file itself
	ProcessSerialNumber		process;		// the finder's psn

	if (!err) err = FSrMakeFSSpec(ref, &fileSpec);

	err = FindAProcess(kFinderSig, kSystemType, &process);
	if (err) goto bailError;
			
	err = AECreateDesc(
		typeProcessSerialNumber, (Ptr)&process, 
		sizeof(process), &myAddressDesc);
	if (err) goto bailError;

	// Create an empty 
	err = AECreateAppleEvent(
		kAEFinderEvents, kAERevealSelection, &myAddressDesc, 
		kAutoGenerateReturnID, kAnyTransactionID, &aeEvent);
	if (err) goto bailError;

	// Make an FSSpec and alias for the parent folder, and an alias for the file
	err = FSMakeFSSpec(fileSpec.vRefNum, fileSpec.parID, NULL, &dirSpec);
	if (err) goto bailError;

	err = NewAlias(NULL, &dirSpec, &dirAlias);
	if (err) goto bailError;

	err = NewAlias(NULL, &fileSpec, &fileAlias);
	if (err) goto bailError;

	/* Create the folder descriptor*/
	HLock((Handle)dirAlias);
	err = AECreateDesc(
		typeAlias, (Ptr)*dirAlias, 
		GetHandleSize((Handle)dirAlias), &aeDirDesc);
	HUnlock((Handle)dirAlias);
	DisposeHandle((Handle)dirAlias);
	if (err) goto bailError;

	err = AEPutParamDesc(&aeEvent,keyDirectObject,&aeDirDesc);
	if (err) goto bailError;

	err = AEDisposeDesc(&aeDirDesc);
	if (err) goto bailError;

	HLock((Handle)fileAlias);
	err = AECreateDesc(
		typeAlias, (Ptr)*fileAlias,
		GetHandleSize((Handle)fileAlias), &listElem);
	HUnlock((Handle)fileAlias);
	DisposeHandle((Handle)fileAlias);
	if (err) goto bailError;

	// Create the file list.
	err = AECreateList(NULL, 0, FALSE, &fileList);
	if (err) goto bailError;

	err = AEPutDesc(&fileList,0,&listElem);
	if (err) goto bailError;

	err = AEDisposeDesc(&listElem);
	if (err) goto bailError;

	err = AEPutParamDesc(&aeEvent,keySelection,&fileList);
	if (err) goto bailError;

	err = AEDisposeDesc(&fileList);
	if (err) goto bailError;

	err = AESend(&aeEvent, NULL,
			kAENoReply+kAEAlwaysInteract+kAECanSwitchLayer,
			kAENormalPriority, kAEDefaultTimeout, NULL, NULL);
	if (err) goto bailError;

	err = AEDisposeDesc(&aeEvent);
	if (err) goto bailError;

	goto done;
	
	bailError:
	//MessageAlertNum(SSLocalize("Some kind of AppleEvent error", "").utf8Z(), err);

	done:
	return err;
}

OSErr	FSrRevealInFinder(const FSRef &ref)	// >>
{
	OSErr					err = noErr;
	OSType				fndr_creator = 'MACS';
	AppleEvent			ae;
	AEDesc				ae_addr_desc;
	AEDesc				ae_par_desc;
	AEDesc				ae_file_desc;
	FSSpec				par_spec, file_spec;
	AliasHandle			par_aliasH = NULL, file_aliasH = NULL;
	bool				is_folderB = FALSE;
	
	err = FSrIsFolder(ref, &is_folderB);
	if (err) goto bail;
	
	structclr(ae);
	structclr(ae_addr_desc);
	structclr(ae_par_desc);
	structclr(ae_file_desc);
	structclr(par_spec);
	structclr(file_spec);

	err = FSrMakeFSSpec(ref, &file_spec);
	if (err) goto bail;

	err = FSpGetParentFolder(&file_spec, &par_spec);
	if (err) goto bail;

	// this section does not throw, and has error reporting at bottom
	
	ERR(AECreateDesc(typeApplSignature, &fndr_creator, sizeof(fndr_creator), &ae_addr_desc));
	
	if (!err) {
		if (is_folderB) {
			ERR(AECreateAppleEvent (kCoreEventClass, kAEOpen,
								&ae_addr_desc, kAutoGenerateReturnID, kAnyTransactionID,
								&ae));
		} else {
			ERR(AECreateAppleEvent(kAEMiscStandards, kAEMakeObjectsVisible, &ae_addr_desc, kAutoGenerateReturnID,
					kAnyTransactionID, &ae));
		}
	}
			
	ERR(NewAlias(NULL, &par_spec, &par_aliasH));
	ERR(NewAlias(NULL, &file_spec, &file_aliasH));
	
	if (!err) {
		HLock((Handle)par_aliasH);
		err = AECreateDesc(typeAlias, *par_aliasH, GetHandleSize((Handle)par_aliasH), &ae_par_desc);
		HUnlock((Handle)par_aliasH);
	}

	ERR(AEPutParamDesc(&ae, keyDirectObject, &ae_par_desc));
	
	if (!err) {
		HLock((Handle)file_aliasH);
		err = AECreateDesc(typeAlias, *file_aliasH, GetHandleSize((Handle)file_aliasH), &ae_file_desc);
		HUnlock((Handle)file_aliasH);
	}

	ERR(AEPutParamDesc(&ae, keyDirectObject, &ae_file_desc));

	ERR(AESend(
		&ae, NULL, kAENoReply + kAECanSwitchLayer, 
		kAENormalPriority, kAEDefaultTimeout, NULL, NULL));
	
	// Get the psn of the Finder and create the target address for the .
	{
		ProcessSerialNumber	psn;

		ERR(FindAProcess('FNDR', 'MACS', &psn));
		ERR(SetFrontProcess(&psn));
	}
	
	bail:
	if (err) {
	//	SuperString		str("Some kind of AppleEvent error");		str.Localize();
	//	MessageAlertNum(str.utf8Z(), err);
	}
	
	if (par_aliasH) DisposeHandle((Handle)par_aliasH);
	if (file_aliasH) DisposeHandle((Handle)file_aliasH);
	
	if (ae_addr_desc.dataHandle) (void)AEDisposeDesc(&ae_addr_desc);
	if (ae_par_desc.dataHandle) (void)AEDisposeDesc(&ae_par_desc);
	if (ae_file_desc.dataHandle) (void)AEDisposeDesc(&ae_file_desc);
	if (ae.dataHandle) (void)AEDisposeDesc(&ae);
	
	return err;
}

class	FSU_ForEach_MakeSafe {
	ustring		&i_newStr;
	public:
	
	FSU_ForEach_MakeSafe(ustring &newStr) : i_newStr(newStr) {}
	
	void	operator ()(UTF8Char ch) {
		if (ch == ':') {
			ch = '-';
		} else if (ch == '/') {
			ch = '\\';
		}
		
		i_newStr.append(1, ch);
	}
};

OSStatus	FSrRename(
	FSRef			*fileRef,
	const UniString	&uniStr)
{
	OSStatus		err	= FSRenameUnicode(
		fileRef, uniStr.i_lengthL, uniStr.i_nameP, 
		kTextEncodingDefaultFormat, fileRef);
	
	return err;
}

OSStatus	FSrUniqueName(
	const FSRef		&parentRef, 
	SuperString		*nameStrP, 
	bool			*changedBP0)
{
	OSStatus		err		= noErr;
	bool			existsB	= true;
	SuperString		extension;

	if (changedBP0) {
		*changedBP0 = false;
	}
	
	nameStrP->pop_ext(&extension);

	do {
		SuperString		temp(*nameStrP);
		
		temp.append(extension);
		
		err = FSrObjectExists(parentRef, temp, &existsB);
			
		if (!err && existsB) {
			IncrementNumberAtEndOfString(nameStrP);
			
			if (changedBP0) {
				*changedBP0 = true;
			}
		}
	} while (!err && existsB);
	
	if (!err) {
		nameStrP->append(extension);
	}
		
	return err;
}

OSStatus	FSrGetFileDates(
	const FSRef		&fileRef, 
	CFAbsoluteTime	*modDateP0,
	CFAbsoluteTime	*creDateP0)
{
	OSStatus		err				= noErr;
	FSCatalogInfo	catInfo;
	FSRefParam		paramBlock;		structclr(paramBlock);
	
	paramBlock.ref			= &fileRef;
	paramBlock.catInfo		= &catInfo;
	paramBlock.whichInfo	= (creDateP0 ? kFSCatInfoCreateDate : 0) 
		| (modDateP0 ? (kFSCatInfoContentMod | kFSCatInfoAttrMod) : 0);
	
	ERR(PBGetCatalogInfoSync(&paramBlock));

	if (!err) {
		
		if (modDateP0) {
			CFAbsoluteTime		contentDate, attribDate;
			
			ERR(UCConvertUTCDateTimeToCFAbsoluteTime(&catInfo.contentModDate, &contentDate));
			ERR(UCConvertUTCDateTimeToCFAbsoluteTime(&catInfo.attributeModDate, &attribDate));
			*modDateP0 = std::max(contentDate, attribDate);
		}

		if (creDateP0) {
			ERR(UCConvertUTCDateTimeToCFAbsoluteTime(&catInfo.createDate, creDateP0));
		}
		
	}
	
	return err;
}

OSStatus		FSrSetFileDates(
	FSRef			&fileRef, 
	CFAbsoluteTime	*modDateP0,
	CFAbsoluteTime	*creDateP0)
{
	OSStatus		err				= noErr;
	FSCatalogInfo	catInfo;
	FSRefParam		paramBlock;		structclr(paramBlock);
	
	paramBlock.ref			= &fileRef;
	paramBlock.catInfo		= &catInfo;
	paramBlock.whichInfo	= (creDateP0 ? kFSCatInfoCreateDate : 0) 
		| (modDateP0 ? (kFSCatInfoContentMod | kFSCatInfoAttrMod) : 0);
	
	if (!err) {
		
		if (modDateP0) {
			ERR(UCConvertCFAbsoluteTimeToUTCDateTime(*modDateP0, &catInfo.contentModDate));

			catInfo.attributeModDate = catInfo.contentModDate;
		}

		if (creDateP0) {
			ERR(UCConvertCFAbsoluteTimeToUTCDateTime(*creDateP0, &catInfo.createDate));
		}
	}
	
	ERR(PBSetCatalogInfoSync(&paramBlock));
	return err;
}

void		CFAbsoluteTimeToDateTimeRec(CFAbsoluteTime &cfTime, DateTimeRec *dtrP)
{
	ScCFReleaser<CFTimeZoneRef>	tz(CFTimeZoneCopyDefault());
	CFGregorianDate				gregDate = CFAbsoluteTimeGetGregorianDate(cfTime, tz);
	
	dtrP->year		= gregDate.year;
	dtrP->month		= gregDate.month;
	dtrP->day		= gregDate.day;
	dtrP->hour		= gregDate.hour;
	dtrP->minute	= gregDate.minute;
	dtrP->second	= gregDate.second;
}

void		DateTimeRecToCFAbsoluteTime(DateTimeRec &dtr, CFAbsoluteTime *cfTimeP)
{
	ScCFReleaser<CFTimeZoneRef>	tz(CFTimeZoneCopyDefault());
	CFGregorianDate				gregDate;
	
	gregDate.year	= dtr.year;
	gregDate.month	= dtr.month;
	gregDate.day	= dtr.day;
	gregDate.hour	= dtr.hour;
	gregDate.minute	= dtr.minute;
	gregDate.second	= dtr.second;
	
	*cfTimeP = CFGregorianDateGetAbsoluteTime(gregDate, tz);
}

OSStatus	FSrGetFileDates(
	const FSRef	&fileRef, 
	DateTimeRec	*modDateP0, 
	DateTimeRec	*creDateP0)
{
	CFAbsoluteTime		creSecs, modSecs;
	OSStatus			err	= noErr;
	
	ERR(FSrGetFileDates(
		fileRef, 
		modDateP0 ? &modSecs : NULL, 
		creDateP0 ? &creSecs : NULL));

	if (!err) {
		if (modDateP0) CFAbsoluteTimeToDateTimeRec(modSecs, modDateP0);
		if (creDateP0) CFAbsoluteTimeToDateTimeRec(creSecs, creDateP0);
	}
	
	return err;
}

OSStatus		FSrSetFileDates(
	FSRef		&fileRef, 
	DateTimeRec	*modDateP0, 
	DateTimeRec	*creDateP0)
{
	OSStatus			err				= noErr;
	CFAbsoluteTime		creSecs, modSecs;

	if (modDateP0) DateTimeRecToCFAbsoluteTime(*modDateP0, &modSecs);
	if (creDateP0) DateTimeRecToCFAbsoluteTime(*creDateP0, &creSecs);

	ERR(FSrSetFileDates(
		fileRef, 
		modDateP0 ? &modSecs : NULL, 
		creDateP0 ? &creSecs : NULL));
	
	return err;
}

OSStatus	FSrTouchFile(FSRef &fileRef, bool touchCreDateB)
{
	CFAbsoluteTime	currentTime = CFAbsoluteTimeGetCurrent();

	return FSrSetFileDates(fileRef, &currentTime, touchCreDateB ? &currentTime : NULL);
}

class FS_ForEach_GetSize {
	UInt64		*i_sizeLP;
	
	public:
	FS_ForEach_GetSize(UInt64 *sizeLP) : i_sizeLP(sizeLP) {}
	
	void	operator()(const FSRef &curRef) {
		OSStatus	err = noErr;
		bool		isFolderB;
		
		ERR(FSrIsFolder(curRef, &isFolderB));
		if (!err) {
			if (isFolderB) {
				FSRefVec		refVec;

				ERR(FSrGetCatInfoBulk(curRef, refVec));
				std::for_each(refVec.begin(), refVec.end(), FS_ForEach_GetSize(i_sizeLP));
			} else {
				UInt64		sizeL;
				
				ERR(FSrGetEOF(curRef, &sizeL));
				*i_sizeLP += sizeL;
			}
		}
		
		if (err) ETX(err);
	}
};

OSStatus		FSrGetFolderSize(const FSRef &folderRef, UInt64 *sizeLP)
{
	OSStatus				err = noErr;
	FS_ForEach_GetSize		fe(sizeLP);
	
	*sizeLP = 0;
	XTE(fe(folderRef));
	return err;
}

class FS_ForEach_Delete {
	bool		i_leaveRootB;
	
	public:	

//	FS_ForEach_Delete() : i_leaveRootB(false) {}
	FS_ForEach_Delete(bool leaveRootB = false) : i_leaveRootB(leaveRootB) {}
	
	void	operator()(const FSRef &curRef) {
		OSStatus	err = noErr;
		bool		isFolderB;
		
		ERR(FSrIsFolder(curRef, &isFolderB));
		if (!err) {
			if (isFolderB) {
				FSRefVec		refVec;

				ERR(FSrGetCatInfoBulk(curRef, refVec));
				std::for_each(refVec.begin(), refVec.end(), FS_ForEach_Delete());
			} else {
				i_leaveRootB = false;
			}

			if (!i_leaveRootB) {
				ERR(FSrDeleteFile(const_cast<FSRef *>(&curRef)));
			}
		}
		
		if (err) ETX(err);
	}
};

OSStatus		FSrDeleteFolder(const FSRef &folderRef, bool leaveRootB)
{
	OSStatus				err = noErr;
	FS_ForEach_Delete		fe(leaveRootB);
	
	XTE(fe(folderRef));
	return err;
}

class FS_ForEach_Delete_Empty {
	bool		i_leaveRootB;
	
	public:	

	FS_ForEach_Delete_Empty() : i_leaveRootB(false) {}
	FS_ForEach_Delete_Empty(bool leaveRootB) : i_leaveRootB(leaveRootB) {}
	
	void	operator()(const FSRef &curRef) {
		OSStatus	err = noErr;
		bool		isFolderB;
		
		ERR(FSrIsFolder(curRef, &isFolderB));
		if (!err) {
			bool			deleteB = !i_leaveRootB;
			std::string		nameStr;
			
			ERR(FSrGetName(curRef, &nameStr));

			if (isFolderB) {
				UInt32			countL;
				FSRefVec		refVec;

				ERR(FSrGetCatInfoBulk(curRef, refVec));
				std::for_each(refVec.begin(), refVec.end(), FS_ForEach_Delete_Empty());

				ERR(FSrCountFilesInFolder(curRef, &countL));
				
				if (countL > 0) {
					deleteB = false;
				}
			} else {
				deleteB = nameStr[0] == '.';
			}

			if (deleteB) {
				ERR(FSrDeleteFile(const_cast<FSRef *>(&curRef), true));
			}
		}
		
		if (err) ETX(err);
	}
};


OSStatus		FSrDeleteEmptyFolders(const FSRef &folderRef, bool leaveRootB)
{
	OSStatus					err = noErr;
	FS_ForEach_Delete_Empty		fe(leaveRootB);
	
	XTE(fe(folderRef));
	return err;
}

OSStatus		FSrExchangeFiles(FSRef *ref1P, FSRef *ref2P)
{
	OSStatus	err = FSExchangeObjects(ref1P, ref2P);
	
	if (err == paramErr) {
		FSRef			ref1par, ref2par;
		SuperString		ref1name, ref2name;
		//	we may be on a DOS volume

		err = noErr;

		ERR(FSrGetParentFolder(*ref1P, &ref1par));
		ERR(FSrGetName(*ref1P, &ref1name));
		
		ERR(FSrGetParentFolder(*ref2P, &ref2par));
		ERR(FSrGetName(*ref2P, &ref2name));
		
		if (ref1name == ref2name) {
			ScCFReleaser<CFStringRef>	fileName(CFStrCreateWithCurAbsTime());
			
			ERR(FSrRename(ref1P, UniString(fileName)));
		}
		
		ERR(FSrMoveFile(ref1par, ref2P));
		ERR(FSrMoveFile(ref2par, ref1P));

		std::swap<FSRef>(*ref1P, *ref2P);

		ERR(FSrRename(ref1P, ref1name));
		ERR(FSrRename(ref2P, ref2name));
		
//		ERR(GetFileTypeAndCreator(
	}
	
	return err;
}
	
OSStatus	CSafeSave::Start()
{
	OSStatus					err = noErr;
	SuperString					nameStr;
	FSVolumeRefNum				vRefNum;

	ERR(FSrGetName(i_file, &nameStr));
	nameStr.prepend("KT_Safe_Save ");

	ERR(FSrGetVRefNum(i_file, &vRefNum));
	ERR(FSrFindFolder(vRefNum, kTemporaryFolderType, &i_safeFile));
	
	{
		ScCFReleaser<CFStringRef>	fileName(CFStrCreateWithCurAbsTime());
		
		ERR(FSrFindOrCreateFolder(i_safeFile, fileName.Get(), &i_parentRef));
	}
	
	ERR(FSrFindOrCreateFile(i_parentRef, nameStr, &i_safeFile, 'TEXT', 'CWIE'));
	
	if (err == noErr) {
		i_origFile = i_safeFile;
	}
	
	return err;
}

OSStatus	CSafeSave::Finish()
{
	OSStatus	err = noErr;
	
	ERR(FSrExchangeFiles(&i_file, &i_safeFile));
	ERR(FSrDeleteFolder(i_parentRef));
	
	i_safeFile = i_file;
	
	if (err) {
		SuperString		name;
		SuperString		str("There was an error saving the file %s is it on a locked volume?");
		
		(void)FSrGetName(i_safeFile, &name);
		str.ssprintf(NULL, name.utf8Z());
		//ReportErr(str.utf8Z(), err);
	}

	return err;
}

bool	GetEditNumber(SuperString *nameStrP, int *numSP)
{
	const std::string	&str(nameStrP->std());
	const char			*ch = &str[str.size() - 5];
	bool				editNameB = isspace(*ch++);
	
	if (editNameB) editNameB = *ch++ == '*';
	if (editNameB) editNameB = isdigit(*ch++);
	if (editNameB) editNameB = isdigit(*ch++);
	if (editNameB) editNameB = isdigit(*ch);
	
	if (editNameB) {
		ch = &str[str.size() - 3];
		
		*numSP = atoi(ch);
		nameStrP->pop_back(5);
	} else {
		*numSP = 0;
	}
	
	return editNameB;
}

//	add a " *001" to the end
void	AddEditNumber(SuperString *strP, char *bufZ0)
{
	SuperString		ext;
	int				num;
	char			bufAC[16];
	bool			is_nakedB = GetEditNumber(strP, &num);

	if (!is_nakedB) {
		strP->pop_ext(&ext);
		GetEditNumber(strP, &num);
	}
	
	if (bufZ0) {
		if (bufZ0[0] == 0) {
			sprintf(bufAC, " *%.3d", ++num);
			strcpy(bufZ0, bufAC);
		} else {
			strcpy(bufAC, bufZ0);
		}
	}
	
	strP->append(bufAC);
	
	if (!is_nakedB) {
		strP->append(ext);
	}
}

OSStatus	FSrDuplicate(FSRef *refP, char *bufZ0)
{
	OSStatus		err = noErr;
	SuperString		nameStr;
	FSRef			folderRef;
	
	ERR(FSrGetName(*refP, &nameStr));
	AddEditNumber(&nameStr, bufZ0);
	ERR(FSrGetParentFolder(*refP, &folderRef));
	ERR(FSrCopyFileToFolder(folderRef, refP, &nameStr));
	
	return err;
}

OSStatus		FSrUpgradeToSharedPrefs(FSRef *fileRefP)
{
	OSStatus	err = noErr;
//	FSRef		folderRef;
	
//	ERR(gApp->i_prefs->Get_kJamsSharedPrefsFolder(&folderRef));
//	ERR(FSrMoveFile(folderRef, fileRefP));
	return err;
}

OSStatus		FSrUpgradeName(
	const FSRef&	folderRef, 
	bool			is_folderB, 
	CFStringRef		newNameStr,
	FSRef			*objectRefP)
{
	OSStatus		err = noErr;
	bool			existsB;

	if (is_folderB) {
		ERR(FSrFolderExists(folderRef, newNameStr, &existsB, objectRefP));
	} else {
		ERR(FSrFileExists(folderRef, newNameStr, &existsB, objectRefP));
	}
	
	if (!existsB) {
		SuperString		oldNameStr(newNameStr);
		
		if (is_folderB) {
			ERR(FSrFolderExists(folderRef, oldNameStr, &existsB, objectRefP, false));
		} else {
			ERR(FSrFileExists(folderRef, oldNameStr, &existsB, objectRefP));
		}
		
		if (existsB) {
			ERR(FSrRename(objectRefP, newNameStr));
			ERR(FSrResolveAlias(objectRefP));
			
			SuperString		name;
			ERR(FSrGetName(*objectRefP, &name));
			
			if (name == oldNameStr) {
				ERR(FSrRename(objectRefP, newNameStr));
			}
		} else {
			if (is_folderB) {
				ERR(FSrFindOrCreateFolder(folderRef, newNameStr, objectRefP));
			} else {
				ERR(FSrFindOrCreateFile(folderRef, newNameStr, objectRefP));
			}
		}
	}
	
	return err;
}

OSStatus		FSrGetResourceFolder(FSRef *fsRefP)
{
	OSStatus		err = fnfErr;
	CFBundleRef     myAppBundle(CFBundleGetMainBundle());

	if ((CFBundleRef)myAppBundle) {
		ScCFReleaser<CFURLRef>	myAppResourcesURL(
			CFBundleCopyResourcesDirectoryURL(myAppBundle));

		if ((CFURLRef)myAppResourcesURL) {
			if (CFURLGetFSRef(myAppResourcesURL, fsRefP)) {
				err = noErr;
			}
		}
	} 

	return err;
}

OSStatus		FSrGetExecutable(FSRef *fsRefP)
{
	OSStatus					err = fnfErr;
	CFBundleRef					myAppBundle(CFBundleGetMainBundle());
	
	if (myAppBundle) {
		ScCFReleaser<CFURLRef>		appURL(CFBundleCopyExecutableURL(myAppBundle));

		if (CFURLGetFSRef(appURL, fsRefP)) {
			err = noErr;
		}
	}
	
	return err;
}

OSStatus	FSrEjectVolume(const FSRef& volRef)
{
	OSStatus				err = noErr;
	bool					volB;
	
	ERR(FSrIsVolume(volRef, &volB));

	if (!volB) {
		err = nsvErr;
	} else {
		FSVolumeOperation		volumeOp;
		FSVolumeRefNum			vRefNum;
		
		ERR(FSrGetVRefNum(volRef, &vRefNum));
		ERR(FSCreateVolumeOperation(&volumeOp));
		if (!err) {
			ERR(FSEjectVolumeAsync(
				vRefNum, 0, volumeOp, NULL, NULL, 
				CFRunLoopGetCurrent(), kCFRunLoopDefaultMode));
			ERR2(FSDisposeVolumeOperation(volumeOp));
		}
	}
	
	return err;
}
