//	kjFs.cpp
#include "CFileRef.h"
#include "CThreads.h"
#include "FSUtils.h"
#include "CHelloFS.h"

static const char  *file_path      = "/hello.txt";
static const char   file_content[] = "Hello World!\n";
static const size_t file_size      = sizeof(file_content)/sizeof(char) - 1;

static int
hello_getattr(const char *path, struct stat *stbuf)
{
    memset(stbuf, 0, sizeof(struct stat));

    if (strcmp(path, "/") == 0) { /* The root directory of our file system. */
        stbuf->st_mode = S_IFDIR | 0755;
        stbuf->st_nlink = 3;
    } else if (strcmp(path, file_path) == 0) { /* The only file we have. */
        stbuf->st_mode = S_IFREG | 0444;
        stbuf->st_nlink = 1;
        stbuf->st_size = file_size;
    } else { /* We reject everything else. */
        return -ENOENT;
    }

    return 0;
}

static int
hello_open(const char *path, struct fuse_file_info *fi)
{
    if (strcmp(path, file_path) != 0) { /* We only recognize one file. */
        return -ENOENT;
    }

    if ((fi->flags & O_ACCMODE) != O_RDONLY) { /* Only reading allowed. */
        return -EACCES;
    }

    return 0;
}

static int
hello_readdir(const char *path, void *buf, fuse_fill_dir_t filler,
              off_t offset, struct fuse_file_info *fi)
{
    if (strcmp(path, "/") != 0) { /* We only recognize the root directory. */
        return -ENOENT;
    }

    filler(buf, ".", NULL, 0);           /* Current directory (.)  */
    filler(buf, "..", NULL, 0);          /* Parent directory (..)  */
    filler(buf, file_path + 1, NULL, 0); /* The only file we have. */

    return 0;
}

static int
hello_read(const char *path, char *buf, size_t size, off_t offset,
           struct fuse_file_info *fi)
{
    if (strcmp(path, file_path) != 0) {
        return -ENOENT;
    }

    if (offset >= file_size) { /* Trying to read past the end of file. */
        return 0;
    }

    if (offset + size > file_size) { /* Trim the read to the file size. */
        size = file_size - offset;
    }

    memcpy(buf, file_content + offset, size); /* Provide the content. */

    return size;
}

/**************************************************/
class CHelloFS_New : CT_Preemptive {
	CHelloFS	*thiz;
	
	public:
	CHelloFS_New(CHelloFS *in_thiz) : CT_Preemptive("HelloFS Thread"), thiz(in_thiz) {
		//	peform any initialization you want here
		
		//	then call to fork the thread. the forked thread will then call operator()
		call();
	}
	
	OSStatus	operator()() {
		return thiz->loop_mt();
	}
};

#define		FS_VOLUMES			"/Volumes"
#define		FS_VOL_NAME			"Hello FS"
#define		FS_MOUNT_POINT		FS_VOLUMES "/" FS_VOL_NAME
#define		FS_ICNS_FILE		"Fuse.icns"

OSStatus	CHelloFS::loop_mt()
{
	//	i know it returns a Unix error, rather than a Mac Error, but so what
	return fuse_main(i_fargs.argc, i_fargs.argv, &i_fs_ops, NULL);
}

CHelloFS::CHelloFS() : i_fuseP(NULL)
{
	OSStatus	err = noErr;
	FSRef		folderRef, execRef;
	CFileRef	volMountFolder(FS_VOLUMES);
	
	structclr(i_fs_ops);
	i_fs_ops.getattr		= hello_getattr;	/* To provide size, permissions, etc. */
	i_fs_ops.open			= hello_open;		/* To enforce read-only access.       */
	i_fs_ops.read			= hello_read;		/* To provide file content.           */
	i_fs_ops.readdir		= hello_readdir;	/* To provide directory listing.      */

	//	create mount point
	ERR(volMountFolder.GetChildFolder(FS_VOL_NAME));
	
	ERR(FSrGetResourceFolder(&folderRef));
	ERR(FSrGetExecutable(&execRef));
	
	if (!err) {
		CFileRef	execFile(execRef);
		CFileRef	iconFile(folderRef);
		
		ERR(iconFile.Descend(FS_ICNS_FILE));

		if (!err) {
			SuperString		str;
			
			i_args.push_back(execFile.GetPath());			//	exec path (must be first)
			i_args.push_back("-ovolname=" FS_VOL_NAME);		//	vol name

			i_args.push_back("-omodules=volicon");			//	volume icon
			str.Set("-ovolicon=");
			str.append(iconFile.GetPath());
			i_args.push_back(str);

			i_args.push_back("-f");							//	foreground
			i_args.push_back("-s");							//	single thread
			i_args.push_back(FS_MOUNT_POINT);				//	mount path
			
			//	now create the argv[] vector
			loop (i_args.size()) {
				i_argsVec.push_back(const_cast<char *>(i_args[_indexS].utf8Z()));	//	Z indicates null terminated char * (in this case unsigned)
			}

			i_fargs	= (fuse_args)FUSE_ARGS_INIT(i_argsVec.size(), &i_argsVec[0]);
		}		
	}
	
	if (!err) {
		new CHelloFS_New(this);
	}
}

CHelloFS::~CHelloFS()
{
	OSStatus	err = noErr;
	FSRef		volRef;
	
	ERR(FSrGetVolume(FS_VOL_NAME, &volRef));
	ERR(FSrEjectVolume(volRef));
}
