#include "FSUtils.h"
#include "CFileRef.h"

CFileRef::CFileRef(const CFileRef& other)
{
	zero();

	if (other.i_fileRefP) {
		operator=(*other.i_fileRefP);
	} else if (other.i_aliasH) {
		Handle		h;
		
		ETX(CopyHandle((Handle)other.i_aliasH, &h))
		i_aliasH = (AliasHandle)h;
	} else if (other.i_pathP) {
		i_pathP = new SuperString(*other.i_pathP);
	}
}

CFileRef::CFileRef(const ustring& path)
{
	zero();
	SetPath(SuperString(path));
}

void		CFileRef::Cleanup(FileRefType type)
{
	if (type == FileRef_NONE) {
		
		if (i_fileRefP) {
			Cleanup(FileRef_ALIAS | FileRef_PATH);
		} else if (i_aliasH) {
			Cleanup(FileRef_PATH);
		}
		
	} else {
		
		if (type & FileRef_FSREF) {
			if (i_fileRefP) {
				delete i_fileRefP;
				i_fileRefP = NULL;
			}
		}

		if (type & FileRef_ALIAS) {
			if (i_aliasH) {
				DisposeHandle((Handle)i_aliasH);
				i_aliasH = NULL; 
			}
		}

		if (type & FileRef_PATH) {
			if (i_pathP) {
				delete i_pathP;
				i_pathP = NULL; 
			}
		}
	}
}

OSStatus				CFileRef::Delete(bool trashFirstB)
{
	OSStatus		err = noErr;
	FSRef			fileRef(*this);
	
	ERR(FSrDeleteFile(&fileRef, trashFirstB));
	if (!err) {
		zero();
	}
	
	return err;
}

OSStatus				CFileRef::Trash()
{
	OSStatus		err = noErr;
	FSRef			fileRef(*this);
	
	ERR(FSrMoveToTrash(&fileRef));
	if (!err) {
		*this = fileRef;
	}
	
	return err;
}

bool			CFileRef::IsFolder()
{
	return false;
}

bool			CFileRef::IsBundle()
{
	return false;
}

OSStatus		CFileRef::Duplicate()
{
	return noErr;
}

CFileRef::operator FSRef&() const {
	if (!i_fileRefP) {
		FSRef		fsRef;
		
		if (i_pathP) {
			ETX(FSrCreateFromPath(i_pathP->utf8(), &fsRef));
		} else {
			Boolean		changedB;
			
			//ASSERT(i_aliasH);
			ETX(FSResolveAlias(NULL, i_aliasH, &fsRef, &changedB));
		}
		
		if (!i_fileRefP) {
			i_fileRefP = new FSRef;
		}
		
		*i_fileRefP = fsRef;
	}

	return *i_fileRefP;
}

bool	CFileRef::operator ==(const FSRef &other)
{
	return FSrEqual(*this, other);
}

bool	CFileRef::operator <(const CFileRef &other)
{
	SuperString		lhs(GetName()), rhs(const_cast<CFileRef&>(other).GetName());
	
	return lhs < rhs;
}


FSRef&	CFileRef::operator =(const FSRef &ref) 
{
	Cleanup(FileRef_ALIAS | FileRef_PATH);

	if (!i_fileRefP) {
		i_fileRefP = new FSRef(ref);
	} else {
		*i_fileRefP = ref;
	}
	
	return *i_fileRefP;
}

const SuperString&		CFileRef::GetPath()
{
	if (!i_pathP) {
		SuperString		path;
		
		ETX(FSrGetPath(*this, &path));
		i_pathP = new SuperString(path);
	}

	return *i_pathP;
}

void				CFileRef::SetPath(const SuperString& path)
{
	Cleanup(FileRef_ALIAS | FileRef_FSREF);

	if (!i_pathP) {
		i_pathP = new SuperString(path);
	} else {
		i_pathP->Set(path);
	}
}

SuperString				CFileRef::GetName()
{
	SuperString			nameStr;
	
	if (i_pathP) {
		
	} else {
		ETX(FSrGetName(*this, &nameStr));
	}
	
	return nameStr;
}

OSStatus				CFileRef::Rename(const SuperString& name)
{
	return noErr;
}

OSStatus				CFileRef::GetParent(FSRef *parRefP)
{
	OSStatus		err = noErr;
	
	ERR(FSrGetParentFolder(*this, parRefP));
	return err;
}

OSStatus				CFileRef::Ascend()
{
	OSStatus		err = noErr;
	FSRef			parentRef;
	
	ERR(GetParent(&parentRef));
	if (!err) {
		XTE(*this = parentRef);
	}

	return err;
}


//	if you don't pass anything for "existsB" then the child will be created
OSStatus				CFileRef::GetChildFolder(const SuperString& child, bool *existsBP0)
{
	OSStatus	err = noErr;
	bool		existsB = false;
	FSRef		fileRef;
	FSRef		folderRef;
	
	XTE(folderRef = *this);
	ERR(FSrFolderExists(folderRef, child, &existsB, &fileRef));
	
	if (!existsB) {
		ERR(FSrFindOrCreateFolder(folderRef, child, &fileRef));
		if (!err) existsB = true;
	}

	if (existsBP0) {
		*existsBP0 = existsB;
	}
	
	if (existsB) {
		XTE(*this = fileRef);
	}
	
	return err;
}

OSStatus				CFileRef::GetChild(
	const SuperString&	child, 
	bool				*existsBP0,
	OSType				fileType,
	OSType				fileCreator)
{
	OSStatus	err = noErr;
	FSRef		fileRef;
	bool		existsB;
	
	ERR(FSrFileExists(*this, child, &existsB, &fileRef));

	if (existsBP0) {
		*existsBP0 = existsB;
	} else {
		//	create the file
	}
	
	if (existsB) {
		XTE(*this = fileRef);
	}
	
	return err;
}

bool					CFileRef::GetRelativeFile(const SuperString& ext)
{
	bool			existsB = false;
	CFileRef		other(*this);
	
	if (other.Ascend() == noErr) {
		SuperString		nameStr(GetName());
		
		nameStr.pop_ext();
		nameStr.append(ext);
		
		if (
			other.GetChild(nameStr, &existsB) == noErr
			&& existsB
		) {
			*this = other;
		}
	}
	
	return existsB;
}

OSStatus				CFileRef::Descend(const SuperString& child)
{
	bool	existsB;
	
	return GetChild(child, &existsB);
}

SuperString				CFileRef::GetExtension()
{
	SuperString		nameStr(GetName());
	SuperString		extStr;
	
	if (!nameStr.empty()) {
		nameStr.pop_ext(&extStr);
	}

	return extStr;
}

OSType					CFileRef::GetExtensionType()
{
	SuperString		extStr	= GetExtension();
	
	return ::GetExtension(extStr.c_str());
}

OSStatus		CFileRef::fopen(const char *accessZ)
{
	OSStatus		err = noErr;

//	ASSERT(i_fileP == NULL);

	ERR_XTE_START {
		const SuperString		&path(GetPath());
		
		i_fileP = ::fopen(path.utf8Z(), accessZ);
	} ERR_XTE_END;

	return (i_fileP == NULL) ? (OSStatus)fnfErr : (OSStatus)noErr;
}

void		CFileRef::fprintf(const char *str,...)
{
	va_list 	args;
	CharVec		charVec;
	
	charVec.resize(16384);

	va_start(args, str);
	
	//	not thread safe!!
	vsprintf(&charVec[0], str, args);
	va_end(args);

	::fprintf(i_fileP, &charVec[0]);
}

OSStatus		CFileRef::fclose()
{
	if (i_fileP) {
		FILE	*closeMeP = i_fileP;
		
		i_fileP = NULL;
		if (::fclose(closeMeP) == EOF) {
			return errno;
		}
	}
	
	return noErr;
}

OSStatus		CFileRef::fread(char *bufP, long offsetL, size_t lengthL)
{
	OSStatus		err = noErr;
	
	ERR(fseek(i_fileP, offsetL, SEEK_SET) == -1 ? errno : noErr);
	ERR(::fread(bufP, 1, lengthL, i_fileP) == lengthL ? noErr : errno);
	return err;
}

SuperString		CFileRef::freadline()
{
	SuperString		str;
	char			chAC[2];
	char&			ch(chAC[0]);
	bool			doneB = false;

	chAC[1] = 0;
	
	do {
		ch = fgetc(i_fileP);
		
		if (IsEOLN(ch)) {
			ch = fgetc(i_fileP);
			if (!IsEOLN(ch)) {
				ungetc(ch, i_fileP);
			}

			doneB = true;
		} else if (ch != EOF) {
			str.append(chAC);
		} else {
			doneB = true;
		}
	} while (!doneB);
	
	return str;
}

//	this fuction can throw!!
UInt64			CFileRef::size()
{
	UInt64		durationL;

	ETX(FSrGetEOF(*this, &durationL));
	return durationL;
}


template<class Function> void for_each(Function f);


/*
template<class Function> void CFileRef::for_each(Function f)
	FSRefVec		&i_refVec;
	
	public: CLZ_ForEach_ScanForSources(FSRefVec *refVecP) : i_refVec(*refVecP) {}
	
	void	operator()(const FSRef &itemRef) {
		bool			is_folderB;

		ETX(FSrIsFolder(itemRef, &is_folderB));
		
		if (is_folderB) {
			FSRefVec	refVec;
			
			ETX(FSrGetCatInfoBulk(itemRef, refVec));
			std::for_each(
				refVec.begin(), refVec.end(), 
				CLZ_ForEach_ScanForSources(&i_refVec));
		} else {
			OSType		ext;
			
			ETX(FSrGetExtensionType(itemRef, &ext));

			if (Is_C_Ext(ext)) {
				if (FileHasLocalize(itemRef)) {
					i_refVec.push_back(itemRef);
				}
			}
		}
	}
};

*/